﻿IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[hips].[PatientInHospitalGet]') AND type in (N'P', N'PC'))
BEGIN
	DROP PROCEDURE [hips].[PatientInHospitalGet]
END
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

-- =============================================
-- Create date:   20 Nov 2013
-- Description:   Gets a list of patients in hospital who have an active IHI and match other given criteria.
-- Modified:
-- 17 March 2014 - Meet SHINeH Requirements: with or without IHI or both, return ConsentWithdrawn, EpisodeCount, AdmissionReason, SourceSystemEpisodeId
-- Modified: 23/03/2015	changed HealthProviderOrganisationNetworkId to return from HealthProviderOrganisation not PatientMasterIhi
-- 04/02/2016 Updated to use EpisodeTypeId instead of EpisodeLifecycleId
-- =============================================
CREATE PROCEDURE [hips].[PatientInHospitalGet] 
(
	@HospitalCodeSystem VARCHAR(20),
	@HospitalId INT = NULL,
	@WithIhi BIT = NULL,
	@WithPcehr BIT = NULL,
	@ExcludeMedicareExclusions BIT,
	@DaysDischarged INT,
	@DaysAfterService INT = 0
)
AS
BEGIN

DECLARE @HealthProviderOrganisationNetworkId INT
SET @HealthProviderOrganisationNetworkId = (
    SELECT hpo.HealthProviderOrganisationNetworkId 
    FROM hips.HealthProviderOrganisation hpo
    INNER JOIN hips.Hospital h 
                ON h.HealthProviderOrganisationId = hpo.HealthProviderOrganisationId
    WHERE h.HospitalId = @HospitalId
)

DECLARE @DischargeDateCutoff DATETIME
SET @DischargeDateCutoff = DATEADD(d, -@DaysDischarged, GETDATE())

DECLARE @DaysAfterServiceCutoff DATETIME
SET @DaysAfterServiceCutoff = DATEADD(d, -@DaysAfterService, GETDATE())

-- Temporary table to store ID of the most recent matching episode for each
-- patient who matches the conditions in the parameters to this stored procedure
CREATE TABLE #MatchingEpisodes (
	 EpisodeId INT NOT NULL
    ,EpisodeCount INT NOT NULL
)

-- In case there are multiple episodes with the same admission date, pick the
-- most recently created episode.
INSERT	#MatchingEpisodes
SELECT	MAX(e.EpisodeId)
       ,MAX(a.EpisodeCount)
FROM hips.Episode e
 INNER JOIN (
	-- Determine the admission date of the most recent episode for each patient who
	-- has a current episode in the required hospital and meets the WithPcehr condition.
	SELECT	e.PatientId,
			MAX(e.AdmissionDate) AS MaxAdmit,
			COUNT(e.EpisodeId) AS EpisodeCount
	FROM	hips.HospitalPatient hp
	 INNER JOIN hips.Episode e
		ON hp.PatientId = e.PatientId
     INNER JOIN hips.Hospital h
	    ON hp.HospitalId = h.HospitalId
	 LEFT JOIN hips.HealthProviderOrganisationPatient hpop
		ON hp.PatientMasterId = hpop.PatientMasterId
		AND h.HealthProviderOrganisationId = hpop.HealthProviderOrganisationId
	WHERE (@HospitalId IS NULL OR hp.HospitalId = @HospitalId)
	  AND (@WithPcehr IS NULL 
			OR   @WithPcehr = '0' AND (    (hpop.PcehrAdvertised = 0 AND hpop.PcehrDisclosed = 0) 
										OR (hpop.HealthProviderOrganisationId IS NULL AND hpop.PatientMasterId IS NULL) 
									  )
			OR   @WithPcehr = '1' AND (hpop.PcehrAdvertised = 1 OR hpop.PcehrDisclosed = 1)
	  )
	AND (
			(e.EpisodeTypeId = 1 AND (e.DischargeDate IS NULL OR e.DischargeDate > @DischargeDateCutoff))
			OR
			(e.EpisodeTypeId != 1 AND (e.AdmissionDate > @DaysAfterServiceCutoff))
		)
	GROUP BY e.PatientId
 ) a 
	ON e.PatientId = a.PatientId
	AND e.AdmissionDate = a.MaxAdmit
WHERE	(
			(e.EpisodeTypeId = 1 AND (e.DischargeDate IS NULL OR e.DischargeDate > @DischargeDateCutoff))
			OR
			(e.EpisodeTypeId != 1 AND (e.AdmissionDate > @DaysAfterServiceCutoff))
		)
GROUP BY e.PatientId


SELECT pm.PatientMasterId, 
		pm.CurrentSexId,
		pm.DateOfBirth,
		pm.MedicareNumber,
		pm.MedicareIrn,
		pm.DvaNumber,
		pmn.FamilyName,
		pmn.GivenNames,
		pmnt.Description AS Title,
		pmns.Description AS Suffix,
		pmi.Ihi,
		pmi.IhiStatusId,
		pmi.IhiRecordStatusId,
		pmi.DateLastValidated AS IhiLastValidated,
		pmi.RegisteredSexId,
		pmi.RegisteredFamilyName,
		pmi.RegisteredGivenName,
		hpo.HealthProviderOrganisationNetworkId,
		hp.HospitalId,
		hp.Mrn,
		hc.Code as HospitalCode,
		h.Name as HospitalName,
		e.Ward,
		e.Room,
		e.Bed,
		e.AdmissionDate,
		e.DischargeDate,
		rp.FamilyName AS ResponsibleProviderFamilyName,
		rp.GivenNames AS ResponsibleProviderGivenNames,
		rpt.Description AS ResponsibleProviderTitle,
		rps.Description AS ResponsibleProviderSuffix,
		hpop.PcehrAdvertised,
		hpop.PcehrDisclosed,
		e.ConsentWithdrawn,
		me.EpisodeCount,
		e.AdmissionReason,
		e.SourceSystemEpisodeId
FROM #MatchingEpisodes me
 INNER JOIN hips.Episode e
	ON me.EpisodeId = e.EpisodeId
 INNER JOIN hips.HospitalPatient hp
	ON e.PatientId = hp.PatientId
 INNER JOIN hips.PatientMaster pm
    ON hp.PatientMasterId = pm.PatientMasterId
 INNER JOIN hips.PatientMasterName pmn
	ON pm.PatientMasterId = pmn.PatientMasterId
	AND pmn.NameTypeId = 2 -- Current Name
 INNER JOIN hips.HospitalCode hc
	ON hp.HospitalId = hc.HospitalId
	AND hc.CodeSystemId = (SELECT CodeSystemId FROM hips.CodeSystem WHERE Code = @HospitalCodeSystem)
 INNER JOIN hips.Hospital h
    ON hp.HospitalId = h.HospitalId
 INNER JOIN hips.HealthProviderOrganisation hpo
	ON h.HealthProviderOrganisationId = hpo.HealthProviderOrganisationId
 INNER JOIN hips.HealthProviderIndividual rp
	ON e.ResponsibleProviderId = rp.HealthProviderIndividualId
 LEFT JOIN hips.Title pmnt
    ON pmn.TitleId = pmnt.TitleId
 LEFT JOIN hips.Suffix pmns
    ON pmn.SuffixId = pmns.SuffixId
 INNER JOIN hips.Title rpt
    ON rp.TitleId = rpt.TitleId
 INNER JOIN hips.Suffix rps
    ON rp.SuffixId = rps.SuffixId
 LEFT JOIN hips.PatientMasterIhi pmi
	ON pm.PatientMasterId = pmi.PatientMasterId
	AND pmi.HealthProviderOrganisationNetworkId = @HealthProviderOrganisationNetworkId
 LEFT JOIN hips.HealthProviderOrganisationPatient hpop
	ON pm.PatientMasterId = hpop.PatientMasterId
	AND h.HealthProviderOrganisationId = hpop.HealthProviderOrganisationId
 WHERE
	(	@WithIhi IS NULL 
		OR (@WithIhi = '0' AND (pmi.PatientMasterId IS NULL OR pmi.Ihi IS NULL OR pmi.Ihi = ''))
		OR (
			@WithIhi = '1' 
			AND pmi.IhiStatusId = 1 -- Active IHI
			AND pmi.IhiRecordStatusId = 1 -- Verified IHI
			AND LEN(pmi.Ihi) = 16 -- Correct Length IHI				
		)
	)
AND (	@ExcludeMedicareExclusions = '0'
		OR pm.MedicareNumber IS NULL
		OR pm.MedicareNumber NOT IN (SELECT MedicareNumber FROM hips.MedicareExclusion)
	)

END
GO
GRANT EXEC ON [hips].[PatientInHospitalGet] TO RunStoredProcedure
GO


